/*
 * Start with a map file and a list of enclosures and generate am
 * initial database.
 */
#include <unistd.h>
#include <stdio.h>
#include <pthread.h>
#include <errno.h>
#include <netinet/in.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "db.h"
#include "libfma.h"
#include "lf_fabric.h"
#include "lf_fabric_db.h"
#include "lf_mapper_map.h"
#include "lf_dflt_error.h"
#include "lf_switch.h"
#include "lf_symbols.h"
#include "lf_xbar32.h"
#include "libmyri.h"

struct lf_fabric *
map_to_fabric(
  struct lf_mapfile *mp)
{
  struct lf_fabric *fp;
  int h;
  int x;

  LF_CALLOC(fp, struct lf_fabric, 1);
  LF_CALLOC(fp->hosts, struct lf_host *, mp->nnic);
  LF_CALLOC(fp->xbars, struct lf_xbar *, mp->nxbar);
  LF_CALLOC(fp->enclosures, struct lf_enclosure *, mp->nxbar);

  /* create all the hosts */
  for (h=0; h<mp->nnic; ++h) {
    struct lf_host *hp;
    struct lf_nic *nicp;

    nicp = mp->nic[h];

    LF_CALLOC(hp, struct lf_host, 1);
    fp->hosts[h] = hp;

    LF_CALLOC(hp->nics, struct lf_nic *, 1);
    LF_CALLOC(hp->hostname, char, 32);
    sprintf(hp->hostname, "host%04d", h);
    hp->num_nics = 1;
    hp->nics[0] = nicp;

    nicp->host = hp;
  }
  fp->num_hosts = h;

  /*
   * Allocate an enclosure for each xbar
   */
  for (x=0; x<mp->nxbar; ++x) {
    struct lf_enclosure *ep;
    lf_string_t enc_name;
    struct lf_linecard *lp;
    struct lf_xbar *xp;
    struct lf_xbar *newxp;
    lf_string_t serial_no;
    int port;

    xp = mp->xbar[x];

    /* allocate the enclosure */
    sprintf(enc_name, "bogusswitch%04d", x);
    ep = lf_allocate_enclosure("GENSW", enc_name);
    if (ep == NULL) LF_ERROR(("Error allocating enclosure"));
    fp->enclosures[x] = ep;

    sprintf(serial_no, "%d", xp->xbar_id);

    /* allocate linecard to hold the xbar */
    lp = lf_allocate_generic_linecard(ep, 0, xp->num_ports, serial_no);
    if (lp == NULL) LF_ERROR(("Error allocating linecard"));

    /*
     * Replace xbar with this one
     */
    newxp = LF_XBAR(lp->xbars[0]);

    /* move all the topo links */
    for (port=0; port<xp->num_ports; ++port) {
      union lf_node *onp;
      int oport;

      onp = xp->topo_ports[port];
      if (onp != NULL) {
	oport = xp->topo_rports[port];

	lf_make_topo_link(onp, oport, LF_NODE(newxp), port);
	lf_make_topo_link(LF_NODE(newxp), port, onp, oport);
      }
    }
    newxp->xbar_id = x;

    /* assign new pointers and free the old one */
    mp->xbar[x] = newxp;
    fp->xbars[x] = newxp;

    lf_free_xbar(xp);
  }

  fp->num_enclosures = x;
  fp->num_xbars = x;

  /*
   * After one pass through all the xbars, they now all have transceivers.
   * They didn't before, but since they do now we can remake all the links
   * through the xcvrs
   */
  for (x=0; x<fp->num_xbars; ++x) {
    struct lf_xbar *xp;
    int p;

    xp = fp->xbars[x];
    for (p=0; p<xp->num_ports; ++p) {
      union lf_node *onp;

      onp = xp->topo_ports[p];
      if (onp != NULL) {
	lf_connect_topo_nodes(LF_NODE(xp), p, onp, xp->topo_rports[p]);
      }
    }
  }

  return fp;

 except:
  exit(1);
  return NULL;
}


int
main(
  int argc,
  char **argv)
{
  char *mapfile;
  int c;
  extern char *optarg;
  struct lf_mapfile *mp;
  struct lf_fabric *fp;
  db_database_ptr_t dbp;
  struct lf_fabric_db *fdp;
  char *fms_run;
  char *db_name;
  int rc;

  lf_init();

  db_name = NULL;
  fms_run = NULL;
  mapfile = NULL;

  /* parse args list */
  while ((c = getopt(argc, argv, "m:R:N:")) != EOF) switch (c) {

  case 'm':	/* map file (input) */
    mapfile = optarg;
    break;
  case 'R':
    fms_run = optarg;
    break;
  case 'N':
    db_name = optarg;
    break;
  default:
    exit(1);
  }

  /* Make sure we got a map file */
  if (mapfile == NULL) {
    fprintf(stderr, "Must specify map file via -m\n");
    exit(1);
  }

  /*
   * If run is not set, try to first get it from environment
   * and use default value if that does not work.  Then do the same
   * for the database name
   */
  if (fms_run == NULL) fms_run = getenv(LF_ENV_FMS_RUN);
  if (fms_run == NULL || *fms_run == '\0') fms_run = LF_DFLT_FMS_RUN;

  if (db_name == NULL) db_name = getenv(LF_ENV_DB_NAME);
  if (db_name == NULL || *db_name == '\0') db_name = LF_DFLT_DB_NAME;

  /* load the mapper file */
  mp = lf_load_mapper_map(mapfile);
  if (mp == NULL) LF_ERROR(("Loading map file", mapfile));

  /* create a fabric */
  fp = map_to_fabric(mp);

  dbp = db_open_database(fms_run, db_name, TRUE);
  if (dbp == NULL) LF_ERROR(("Error opening database"));

  fdp = lf_fabric_db_handle(dbp);
  if (fdp == NULL) LF_ERROR(("Error getting fabric DB handle"));

  /* load and throw away any existing fabric */
  (void) lf_load_fabric(fdp);

  /* clear everything out of any existing database */
  rc = lf_clear_fabric_db(fdp);
  if (rc != 0) LF_ERROR(("Error clearing fabric DB"));

  rc = lf_add_fabric_to_db(fdp, fp);
  if (rc != 0) LF_ERROR(("Error adding fabric to DB"));

  rc = lf_flush_fabric_db(fdp);
  if (rc != 0) LF_ERROR(("Error writing fabric DB"));

  /* close the tables and database */
  lf_close_fabric_db(fdp);

  exit(0);

 except:
  exit(1);
}
